<?php
/**
 * Theme functions and definitions
 *
 * @package DestinationPalestine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

define( 'DESTINATION_PALESTINE_VERSION', '2.7.1' );

require get_template_directory() . '/includes/ocdi/functions.php';


if ( ! isset( $content_width ) ) {
	$content_width = 800; // Pixels.
}

if ( ! function_exists( 'destination_palestine_setup' ) ) {
	/**
	 * Set up theme support.
	 *
	 * @return void
	 */
	function destination_palestine_setup() {
		if ( is_admin() ) {
			destination_maybe_update_theme_version_in_db();
		}

		if ( apply_filters( 'destination_palestine_register_menus', true ) ) {
			register_nav_menus( [ 'menu-1' => esc_html__( 'Header', 'destination-palestine' ) ] );
			register_nav_menus( [ 'menu-2' => esc_html__( 'Footer', 'destination-palestine' ) ] );
		}

		if ( apply_filters( 'destination_palestine_post_type_support', true ) ) {
			add_post_type_support( 'page', 'excerpt' );
		}

		if ( apply_filters( 'destination_palestine_add_theme_support', true ) ) {
			add_theme_support( 'post-thumbnails' );
			add_theme_support( 'automatic-feed-links' );
			add_theme_support( 'title-tag' );
			add_theme_support(
				'html5',
				[
					'search-form',
					'comment-form',
					'comment-list',
					'gallery',
					'caption',
					'script',
					'style',
				]
			);
			add_theme_support(
				'custom-logo',
				[
					'height'      => 100,
					'width'       => 350,
					'flex-height' => true,
					'flex-width'  => true,
				]
			);

			/*
			 * Editor Style.
			 */
			add_editor_style( 'classic-editor.css' );

			/*
			 * Gutenberg wide images.
			 */
			add_theme_support( 'align-wide' );

			/*
			 * WooCommerce.
			 */
			if ( apply_filters( 'destination_palestine_add_woocommerce_support', true ) ) {
				// WooCommerce in general.
				add_theme_support( 'woocommerce' );
				// Enabling WooCommerce product gallery features (are off by default since WC 3.0.0).
				// zoom.
				add_theme_support( 'wc-product-gallery-zoom' );
				// lightbox.
				add_theme_support( 'wc-product-gallery-lightbox' );
				// swipe.
				add_theme_support( 'wc-product-gallery-slider' );
			}
		}
	}
}
add_action( 'after_setup_theme', 'destination_palestine_setup' );

function destination_maybe_update_theme_version_in_db() {
	$theme_version_option_name = 'destination_theme_version';
	// The theme version saved in the database.
	$destination_theme_db_version = get_option( $theme_version_option_name );

	// If the 'destination_theme_version' option does not exist in the DB, or the version needs to be updated, do the update.
	if ( ! $destination_theme_db_version || version_compare( $destination_theme_db_version, DESTINATION_PALESTINE_VERSION, '<' ) ) {
		update_option( $theme_version_option_name, DESTINATION_PALESTINE_VERSION );
	}
}

if ( ! function_exists( 'destination_palestine_scripts_styles' ) ) {
	/**
	 * Theme Scripts & Styles.
	 *
	 * @return void
	 */
	function destination_palestine_scripts_styles() {
		$min_suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		if ( apply_filters( 'destination_palestine_enqueue_style', true ) ) {
			wp_enqueue_style(
				'destination-palestine',
				get_template_directory_uri() . '/style' . $min_suffix . '.css',
				[],
				DESTINATION_PALESTINE_VERSION
			);
		}

		if ( apply_filters( 'destination_palestine_enqueue_theme_style', true ) ) {
			wp_enqueue_style(
				'destination-palestine-theme-style',
				get_template_directory_uri() . '/theme' . $min_suffix . '.css',
				[],
				DESTINATION_PALESTINE_VERSION
			);
		}

		wp_enqueue_style(
			'destination-palestine-custom',
			get_template_directory_uri() . '/custom.css',
			[],
			DESTINATION_PALESTINE_VERSION
		);

	}
}
add_action( 'wp_enqueue_scripts', 'destination_palestine_scripts_styles' );

if ( ! function_exists( 'destination_palestine_register_elementor_locations' ) ) {
	/**
	 * Register Elementor Locations.
	 *
	 * @param ElementorPro\Modules\ThemeBuilder\Classes\Locations_Manager $elementor_theme_manager theme manager.
	 *
	 * @return void
	 */
	function destination_palestine_register_elementor_locations( $elementor_theme_manager ) {
		if ( apply_filters( 'destination_palestine_register_elementor_locations', true ) ) {
			$elementor_theme_manager->register_all_core_location();
		}
	}
}
add_action( 'elementor/theme/register_locations', 'destination_palestine_register_elementor_locations' );

if ( ! function_exists( 'destination_palestine_content_width' ) ) {
	/**
	 * Set default content width.
	 *
	 * @return void
	 */
	function destination_palestine_content_width() {
		$GLOBALS['content_width'] = apply_filters( 'destination_palestine_content_width', 800 );
	}
}
add_action( 'after_setup_theme', 'destination_palestine_content_width', 0 );

if ( is_admin() ) {
	require get_template_directory() . '/includes/admin-functions.php';
}

/**
 * If Elementor is installed and active, we can load the Elementor-specific Settings & Features
*/

// Allow active/inactive via the Experiments
require get_template_directory() . '/includes/elementor-functions.php';

/**
 * Include customizer registration functions
*/
function destination_register_customizer_functions() {
	if ( is_customize_preview() ) {
		require get_template_directory() . '/includes/customizer-functions.php';
	}
}
add_action( 'init', 'destination_register_customizer_functions' );

if ( ! function_exists( 'destination_palestine_check_hide_title' ) ) {
	/**
	 * Check hide title.
	 *
	 * @param bool $val default value.
	 *
	 * @return bool
	 */
	function destination_palestine_check_hide_title( $val ) {
		if ( defined( 'ELEMENTOR_VERSION' ) ) {
			$current_doc = Elementor\Plugin::instance()->documents->get( get_the_ID() );
			if ( $current_doc && 'yes' === $current_doc->get_settings( 'hide_title' ) ) {
				$val = false;
			}
		}
		return $val;
	}
}
add_filter( 'destination_palestine_page_title', 'destination_palestine_check_hide_title' );

/**
 * BC:
 * In v2.7.0 the theme removed the `destination_palestine_body_open()` from `header.php` replacing it with `wp_body_open()`.
 * The following code prevents fatal errors in child themes that still use this function.
 */
if ( ! function_exists( 'destination_palestine_body_open' ) ) {
	function destination_palestine_body_open() {
		wp_body_open();
	}
}


function dp_custom_sidebar() {
	register_sidebar(
		array (
			'name' => __( 'Listing Sidebar', 'destination-palestine' ),
			'id' => 'listing-side-bar',
			'description' => __( 'Sidebar used on listings page.', 'destination-palestine' ),
			'before_widget' => '<div class="widget-content">',
			'after_widget' => "</div>",
			'before_title' => '<h3 class="widget-title">',
			'after_title' => '</h3>',
		)
	);
}
add_action( 'widgets_init', 'dp_custom_sidebar' );



/**
 * Add the content type to job type.
 *
 * @since DestinationPalestine 1.0
 *
 */
function dp_post_after_save( $params ) {
	// 'post_id'
	// 'post_uuid'
	// 'wp_lang'
	// 'trans_wp_lang'
	// 'content_type'
	// 'content_type_name'
	extract($params);

	$wp_category = 'job_listing_type';

	update_post_meta( $post_id, 'main_category', $content_type_name );

}
add_action( 'dp_post_after_save', 'dp_post_after_save', 10, 4 );


/**
 * Modify Destination Palestine address.
 *
 * @since Findpal 1.0
 *
 * @return boolean 	Modified the state.
 */
function dp_modify_address( $value, $meta_key, $post_id, $wp_languages ) {

	$governorate = null;
	if(isset($value['administrative_area'])) {
		$governorate = get_governorate($value['administrative_area']);
		update_post_meta( $post_id, 'governorate', $governorate );
	}


	return $value;
}
add_filter( 'dp_modify_value_address', 'dp_modify_address', 10, 4 );


/**
 * Modify Destination Palestine GPS.
 *
 * @since Findpal 1.0
 *
 * @return boolean 	Modified the state.
 */
function dp_modify_gps_coordinates( $value, $meta_key, $post_id, $wp_languages ) {

	return implode(',', $value);
}
add_filter( 'dp_modify_value_gps_coordinates', 'dp_modify_gps_coordinates', 10, 4 );

function get_governorate($key) {
	$list = [
		'PS-BTH' => 'Bethlehem',
		'PS-DEB' => 'Deir El Balah',
		'PS-GZA' => 'Gaza',
		'PS-HBN' => 'Hebron',
		'PS-JEN' => 'Jenin',
		'PS-JRH' => 'Jericho and Al Aghwar',
		'PS-JEM' => 'Jerusalem',
		'PS-KYS' => 'Khan Yunis',
		'PS-NBS' => 'Nablus',
		'PS-NGZ' => 'North Gaza',
		'PS-QQA' => 'Qalqilya',
		'PS-RFH' => 'Rafah',
		'PS-RBH' => 'Ramallah',
		'PS-SLT' => 'Salfit',
		'PS-TBS' => 'Tubas',
		'PS-TKM' => 'Tulkarm'
	];

	return $list[$key] ?? false;
}


/**
 * Include the TGM_Plugin_Activation class.
 */
require_once dirname( __FILE__ ) . '/vendors/class-tgm-plugin-activation.php';

add_action( 'tgmpa_register', 'dp_theme_register_required_plugins' );

/**
 * Register the required plugins for this theme.
 *
 * This function is hooked into tgmpa_init, which is fired within the
 * TGM_Plugin_Activation class constructor.
 */
function dp_theme_register_required_plugins() {
	/*
	 * Array of plugin arrays. Required keys are name and slug.
	 * If the source is NOT from the .org repo, then source is also required.
	 */
	$plugins = array(

		// This is an example of how to include a plugin bundled with a theme.
		array(
			'name'               => 'Pro Elements', // The plugin name.
			'slug'               => 'pro-elements', // The plugin slug (typically the folder name).
			'source'             => get_stylesheet_directory() . '/vendors/plugins/pro-elements.zip', // The plugin source.
			'required'           => true, // If false, the plugin is only 'recommended' instead of required.
			'force_activation'   => false, // If true, plugin is activated upon theme activation and cannot be deactivated until theme switch.
			'force_deactivation' => false, // If true, plugin is deactivated upon theme switch, useful for theme-specific plugins.
			'external_url'       => 'https://proelements.org/', // If set, overrides default API URL and points to an external URL.
		),

		array(
			'name'                     => esc_html__( 'Elementor Page Builder', 'destination-palestine' ),
		    'slug'                     => 'elementor',
		    'required'                 => true,
		),

		array(
			'name'                     => esc_html__( 'Palestinian Tourism Media Library', 'destination-palestine' ),
	        'slug'                     => 'ptml-images',
	        'required'                 => true ,
	        'source'				   => 'https://downloads.destinationpalestine.io/plugins/ptml-images.zip'
		),
	
		array(
			'name'                     => esc_html__( 'Destination Palestine', 'destination-palestine' ),
			'slug'                     => 'destination-palestine',
			'required'                 => true ,
			'source'				   => 'https://downloads.destinationpalestine.io/plugins/destination-palestine.zip'
		),
		array(
	        'name'                  => esc_html__( 'One Click Demo Import', 'destination-palestine' ),
	        'slug'                  => 'one-click-demo-import',
	        'required'              => false,
	        'force_activation'      => false,
	        'force_deactivation'    => false,
	        'external_url'          => '',
    	),
		array(
	        'name'                  => esc_html__( 'Dynamic Conditions', 'destination-palestine' ),
	        'slug'                  => 'dynamicconditions',
	        'required'              => true,
	        'force_activation'      => false,
	        'force_deactivation'    => false,
	        'external_url'          => '',
    	),
		array(
	        'name'                  => esc_html__( 'Filter Everything', 'destination-palestine' ),
	        'slug'                  => 'filter-everything',
	        'required'              => true,
	        'force_activation'      => true,
	        'force_deactivation'    => false,
	        'external_url'          => '',
    	)
	

	);
	tgmpa( $plugins );
}

function dp_setup_options() {
    
    $dp_defaults = array(
        "cleanup_db_on_plugin_uninstall" => false,
        "single_post_type" => true,
        "use_minimal" => false,
        "show_labels" => true
    );

    $dp_settings = get_option('destination_palestine_settings', $dp_defaults);
    $dp_settings['single_post_type'] = true;
    update_option('destination_palestine_settings', $dp_settings);
}
add_action('after_switch_theme', 'dp_setup_options');


function allow_zip_uploads ( $existing_mimes=array() ) { 
	$existing_mimes['zip'] = 'application/zip';
	$existing_mimes['gz'] = 'application/x-gzip';
	return $existing_mimes;
}
add_filter('upload_mimes', 'allow_zip_uploads'); 